import asyncio
import websockets
import json
from typing import Set
from websockets.asyncio.server import ServerConnection  # Import đúng cho ServerConnection
from websockets.protocol import State  # Import State để check connection open

# Lưu tất cả clients kết nối (để chọn 1 client cụ thể)
connected_clients: Set[ServerConnection] = set()  # Sử dụng type đúng

# Hàm gửi countdown từ 200 đến 1 đến 1 client cụ thể (real-time với delay)
async def send_countdown_to_one_client(target_websocket: ServerConnection):
    for i in range(200, 0, -1):  # Vòng lặp for từ 200 xuống 1
        if target_websocket.state is State.OPEN:  # Kiểm tra client còn kết nối
            try:
                await target_websocket.send(json.dumps({
                    "status": "200",
                    "countdown": i,
                    "message": f"Countdown real-time: {i}"
                }))
                print(f"Đã gửi {i} đến client")
            except websockets.exceptions.ConnectionClosed:
                print("Client đã disconnect trong countdown, dừng")
                break
        else:
            print("Client đã đóng, dừng countdown")
            break
        await asyncio.sleep(0.1)  # Delay 0.1s để real-time mượt mà (tổng ~20s cho 200 lần)

# Hàm handler cho mỗi client kết nối
async def real_time_handler(websocket: ServerConnection):
    # Thêm client mới vào set
    connected_clients.add(websocket)
    print(f"Client mới kết nối. Tổng clients: {len(connected_clients)}")
    
    # Gửi ngay message "200" đầu tiên đến CHỈ client này
    try:
        await websocket.send(json.dumps({
            "status": "200",
            "message": "OK - Bắt đầu countdown từ 200 đến 1 (real-time)"
        }))
        print("Đã gửi '200' khởi đầu đến client mới")
    except websockets.exceptions.ConnectionClosed:
        print("Lỗi gửi ngay khi connect")
        return
    
    # Bắt đầu countdown đến client này (1 client cụ thể)
    countdown_task = asyncio.create_task(send_countdown_to_one_client(websocket))
    
    try:
        # Vòng lặp nhận message từ client (bidirectional real-time)
        async for message in websocket:
            print(f"Nhận từ client: {message}")
            
            # Nếu client gửi "stop", hủy countdown
            try:
                data = json.loads(message)
                if data.get("action") == "stop":
                    countdown_task.cancel()
                    print("Dừng countdown theo yêu cầu")
                    await websocket.send(json.dumps({"status": "200", "message": "OK - Đã dừng countdown"}))
            except json.JSONDecodeError:
                pass  # Xử lý như text thường
    
    except websockets.exceptions.ConnectionClosed:
        print("Client disconnect")
        countdown_task.cancel()  # Hủy task nếu disconnect
    finally:
        # Xóa client khi disconnect
        connected_clients.discard(websocket)
        print(f"Client rời đi. Tổng clients: {len(connected_clients)}")

# Timer global (tùy chọn, gửi update mỗi 10s đến tất cả)
async def global_timer():
    while True:
        await asyncio.sleep(10.0)
        if connected_clients:
            disconnected = set()
            for client in list(connected_clients):
                if client.state is State.OPEN:  # Kiểm tra đúng cách
                    try:
                        await client.send(json.dumps({
                            "status": "200",
                            "message": "OK - Update real-time từ server timer"
                        }))
                    except websockets.exceptions.ConnectionClosed:
                        disconnected.add(client)
            # Xóa clients đã disconnect
            connected_clients -= disconnected
            print("Đã gửi update từ timer đến tất cả clients")

async def main():
    # Tạo task timer chạy song song (tùy chọn)
    timer_task = asyncio.create_task(global_timer())
    
    # Khởi động server tại ws://localhost:8765
    async with websockets.serve(real_time_handler, "localhost", 8765):
        print("Real-time Countdown Server chạy tại ws://localhost:8765")
        print("Mỗi client mới sẽ nhận countdown từ 200 đến 1 (for loop real-time)!")
        # Chạy timer và server mãi mãi
        await asyncio.gather(timer_task, asyncio.Future())

if __name__ == "__main__":
    asyncio.run(main())