<?php
// function.php - Nâng cấp với API Location chính xác nhất
// Code bởi Vincenzo Cassano - Premium IP Geolocation

// Include config (use absolute path to avoid errors)
include $_SERVER['DOCUMENT_ROOT'] . '/config/config.php';

// Instantiate and connect DB (add error handling)
$_vincenzo = new Vincenzo();
$_vincenzo->connect();

// Helper functions (with error checks)
if (!function_exists('BASE_URL')) {
    function BASE_URL($url = '') {
        $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://";
        $host = $_SERVER["HTTP_HOST"];
        $base = $protocol . $host . '/';
        return $base . ltrim($url, '/');
    }
}

if (!function_exists('format_money')) {
    function format_money($number, $decimals = 0, $dec_point = '.', $thousands_sep = ',') {
        if (!is_numeric($number)) return '0';
        return number_format((float)$number, $decimals, $dec_point, $thousands_sep);
    }
}

if (!function_exists('status_members')) {
    function status_members($status) {
        if ($status == 1) {
            return '<span class="badge bg-success">Hoạt Động</span>';
        } else {
            return '<span class="badge bg-danger">Banned</span>';
        }
    }
}

if (!function_exists('status_ruttien')) {
    function status_ruttien($status) {
        $statuses = [
            "Thành Công" => '<span class="badge bg-success">Thành Công</span>',
            "Thất Bại" => '<span class="badge bg-danger">Thất Bại</span>',
            "Hoàn" => '<span class="badge bg-warning">Hoàn</span>',
            "Đang Duyệt" => '<span class="badge bg-info">Đang Duyệt</span>'
        ];
        return $statuses[$status] ?? '<span class="badge bg-secondary">Unknown</span>';
    }
}

// =========================================================================
// HÀM LẤY LOCATION TỪ IP - NÂNG CẤP PREMIUM
// =========================================================================

/**
 * Lấy thông tin địa lý từ IP - Sử dụng các API đáng tin cậy nhất
 * 
 * @param string $ip Địa chỉ IP cần tra cứu
 * @return string|null Chuỗi địa điểm hoặc null nếu thất bại
 */
if (!function_exists('getLocationFromIP')) {
    function getLocationFromIP($ip) {
        // Kiểm tra IP hợp lệ và không phải IP local/private
        if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
            return null;
        }

        // --- ƯU TIÊN 1: ip-api.com (Miễn phí, chính xác cao, 45 requests/phút) ---
        $url1 = "http://ip-api.com/json/{$ip}?fields=status,message,country,countryCode,region,regionName,city,zip,lat,lon,timezone,isp,org,as";
        $response1 = curl_get_data($url1);
        if ($response1) {
            $data1 = json_decode($response1);
            if (isset($data1->status) && $data1->status === 'success') {
                $location = [];
                if (!empty($data1->city)) $location[] = $data1->city;
                if (!empty($data1->regionName)) $location[] = $data1->regionName;
                if (!empty($data1->country)) $location[] = $data1->country;
                
                if (!empty($location)) {
                    return implode(', ', $location);
                }
            }
        }

        // --- DỰ PHÒNG 2: ipapi.co (Miễn phí 1000 requests/ngày, rất chính xác) ---
        $url2 = "https://ipapi.co/{$ip}/json/";
        $response2 = curl_get_data($url2);
        if ($response2) {
            $data2 = json_decode($response2);
            if (isset($data2->city) && !isset($data2->error)) {
                $location = [];
                if (!empty($data2->city)) $location[] = $data2->city;
                if (!empty($data2->region)) $location[] = $data2->region;
                if (!empty($data2->country_name)) $location[] = $data2->country_name;
                
                if (!empty($location)) {
                    return implode(', ', $location);
                }
            }
        }

        // --- DỰ PHÒNG 3: ipwhois.app (Miễn phí 10000 requests/tháng) ---
        $url3 = "http://ipwhois.app/json/{$ip}?objects=city,region,country";
        $response3 = curl_get_data($url3);
        if ($response3) {
            $data3 = json_decode($response3);
            if (isset($data3->success) && $data3->success === true) {
                $location = [];
                if (!empty($data3->city)) $location[] = $data3->city;
                if (!empty($data3->region)) $location[] = $data3->region;
                if (!empty($data3->country)) $location[] = $data3->country;
                
                if (!empty($location)) {
                    return implode(', ', $location);
                }
            }
        }

        // --- DỰ PHÒNG 4: ipinfo.io (Miễn phí 50000 requests/tháng) ---
        $url4 = "https://ipinfo.io/{$ip}/json";
        $response4 = curl_get_data($url4);
        if ($response4) {
            $data4 = json_decode($response4);
            if (isset($data4->city) && !isset($data4->error)) {
                $location = [];
                if (!empty($data4->city)) $location[] = $data4->city;
                if (!empty($data4->region)) $location[] = $data4->region;
                if (!empty($data4->country)) $location[] = $data4->country;
                
                if (!empty($location)) {
                    return implode(', ', $location);
                }
            }
        }

        return null; // Tất cả đều thất bại
    }
}

/**
 * Hàm hỗ trợ thực hiện cuộc gọi API bằng cURL với timeout và retry
 * 
 * @param string $url URL của API
 * @param int $timeout Thời gian timeout (giây)
 * @param int $retries Số lần thử lại
 * @return string|false Dữ liệu trả về hoặc false nếu lỗi
 */
if (!function_exists('curl_get_data')) {
    function curl_get_data($url, $timeout = 5, $retries = 1) {
        $attempt = 0;
        
        while ($attempt < $retries) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 3);
            curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Bỏ qua SSL nếu cần
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            
            $data = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            curl_close($ch);

            // Kiểm tra thành công
            if ($http_code >= 200 && $http_code < 300 && $data !== false) {
                return $data;
            }

            $attempt++;
            if ($attempt < $retries) {
                usleep(200000); // Chờ 0.2 giây trước khi thử lại
            }
        }

        return false;
    }
}

/**
 * Cache location để tránh gọi API nhiều lần cho cùng IP
 * 
 * @param string $ip Địa chỉ IP
 * @return string|null Location từ cache hoặc API
 */
if (!function_exists('getLocationFromIP_Cached')) {
    function getLocationFromIP_Cached($ip) {
        static $cache = [];
        
        // Kiểm tra cache trong memory
        if (isset($cache[$ip])) {
            return $cache[$ip];
        }
        
        // Gọi API
        $location = getLocationFromIP($ip);
        
        // Lưu vào cache
        if ($location !== null) {
            $cache[$ip] = $location;
        }
        
        return $location;
    }
}

// =========================================================================
// KẾT THÚC PHẦN NÂNG CẤP
// =========================================================================

// Additional helpers
if (!function_exists('parse_vietnamese_number')) {
    function parse_vietnamese_number($str) {
        if (empty($str)) return 0;
        $clean = str_replace('.', '', $str);
        $clean = str_replace(',', '.', $clean);
        return (int) $clean;
    }
}

// Test connection (remove in production)
if (isset($_GET['test'])) {
    echo "<h2>Functions Test</h2>";
    echo "BASE_URL: " . BASE_URL('test') . "<br>";
    echo "Format money: " . format_money(59000) . "<br>";
    echo "Status members (1): " . status_members(1) . "<br>";
    echo "Status ruttien (Thành Công): " . status_ruttien("Thành Công") . "<br>";
    echo "Parse Vietnamese (59.000): " . parse_vietnamese_number("59.000") . "<br>";
    
    echo "<hr><h3>IP Location Test</h3>";
    $test_ip = $_SERVER['REMOTE_ADDR'] ?? '8.8.8.8';
    echo "Testing IP: {$test_ip}<br>";
    echo "Location: " . (getLocationFromIP_Cached($test_ip) ?? 'Not found') . "<br>";
    exit;
}
?>